import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, CheckCircle, Camera, GraduationCap, Sparkles, Award, Clock, BookOpen, User, Star, Users, Shield } from 'lucide-react'
import { motion } from 'framer-motion'
import './Enroll.css'

const Enroll = () => {
  const navigate = useNavigate()
  const [selectedLevel, setSelectedLevel] = useState('')
  const [startDate, setStartDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const courseLevels = [
    { 
      id: 'beginner', 
      name: 'Начинающий', 
      icon: GraduationCap,
      color: '#2563EB',
      description: 'Основы фотографии для новичков',
      duration: '8 недель',
      lessons: '24 урока'
    },
    { 
      id: 'advanced', 
      name: 'Продвинутый', 
      icon: Sparkles,
      color: '#3B82F6',
      description: 'Углубленное изучение техник',
      duration: '12 недель',
      lessons: '36 уроков'
    },
    { 
      id: 'professional', 
      name: 'Профессиональный', 
      icon: Award,
      color: '#60A5FA',
      description: 'Мастерство для профессионалов',
      duration: '16 недель',
      lessons: '48 уроков'
    },
    { 
      id: 'retouch', 
      name: 'Ретушь', 
      icon: Camera,
      color: '#2563EB',
      description: 'Обработка и ретушь фотографий',
      duration: '6 недель',
      lessons: '18 уроков'
    }
  ]

  const instructors = [
    { name: 'Анна Петрова', experience: '12 лет', rating: 5, specialty: 'Портретная съемка', students: '5000+' },
    { name: 'Михаил Соколов', experience: '15 лет', rating: 5, specialty: 'Пейзажная фотография', students: '7000+' },
    { name: 'Елена Волкова', experience: '10 лет', rating: 5, specialty: 'Студийная съемка', students: '4000+' },
    { name: 'Дмитрий Иванов', experience: '18 лет', rating: 5, specialty: 'Ретушь и обработка', students: '9000+' }
  ]

  const guarantees = [
    { icon: Shield, title: 'Гарантия качества', text: 'Возврат средств при неудовлетворенности' },
    { icon: Clock, title: 'Гибкий график', text: 'Учитесь в удобное время' },
    { icon: Users, title: 'Поддержка', text: 'Помощь на всех этапах обучения' },
    { icon: Award, title: 'Сертификат', text: 'Официальный документ по окончании' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedLevel && startDate) {
      const savedCourses = localStorage.getItem('photoCourses')
      const courses = savedCourses ? JSON.parse(savedCourses) : []
      
      const levelData = courseLevels.find(l => l.id === selectedLevel)
      const randomInstructor = instructors[Math.floor(Math.random() * instructors.length)]
      
      const newCourse = {
        id: Date.now(),
        module: levelData.name,
        instructor: randomInstructor.name,
        completionDate: new Date(new Date(startDate).getTime() + parseInt(levelData.duration) * 7 * 24 * 60 * 60 * 1000).toISOString().split('T')[0],
        rating: 0
      }

      courses.push(newCourse)
      localStorage.setItem('photoCourses', JSON.stringify(courses))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedLevel('')
        setStartDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="enroll-page">
      <div className="container">
        <motion.section 
          className="enroll-hero"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
        >
          <h1 className="enroll-title">Записаться на курс</h1>
          <p className="enroll-subtitle">
            Выберите уровень обучения и дату старта
          </p>
        </motion.section>

        <motion.section 
          className="enrollment-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.1 }}
        >
          <div className="enrollment-form-wrapper">
            <form className="enrollment-form" onSubmit={handleSubmit}>
              <div className="form-group">
                <label className="form-label">
                  Выберите уровень курса
                </label>
                <div className="levels-list">
                  {courseLevels.map((level) => {
                    const IconComponent = level.icon
                    return (
                      <motion.button
                        key={level.id}
                        type="button"
                        className={`level-item ${selectedLevel === level.id ? 'selected' : ''}`}
                        onClick={() => setSelectedLevel(level.id)}
                        whileHover={{ x: 3, transition: { duration: 0.2 } }}
                        whileTap={{ scale: 0.98 }}
                      >
                        <div className="level-icon">
                          <IconComponent size={28} strokeWidth={2} color={level.color} />
                        </div>
                        <div className="level-content">
                          <h3 className="level-name">{level.name}</h3>
                          <p className="level-description">{level.description}</p>
                          <div className="level-meta">
                            <span className="level-duration">
                              <Clock size={16} strokeWidth={2} />
                              {level.duration}
                            </span>
                            <span className="level-lessons">
                              <BookOpen size={16} strokeWidth={2} />
                              {level.lessons}
                            </span>
                          </div>
                        </div>
                        {selectedLevel === level.id && (
                          <motion.div 
                            className="level-check"
                            initial={{ scale: 0 }}
                            animate={{ scale: 1 }}
                          >
                            <CheckCircle size={24} strokeWidth={2} color="#2563EB" />
                          </motion.div>
                        )}
                      </motion.button>
                    )
                  })}
                </div>
              </div>

              <div className="form-group">
                <label htmlFor="startDate" className="form-label">
                  Дата старта курса
                </label>
                <input
                  type="date"
                  id="startDate"
                  className="form-input"
                  value={startDate}
                  onChange={(e) => setStartDate(e.target.value)}
                  min={minDate}
                  max={maxDate}
                  required
                />
              </div>

              <motion.button 
                type="submit" 
                className="submit-button" 
                disabled={!selectedLevel || !startDate}
                whileHover={{ scale: 1.02 }}
                whileTap={{ scale: 0.98 }}
              >
                <span>Записаться на курс</span>
                <CheckCircle size={20} strokeWidth={2} className="btn-icon" />
              </motion.button>

              {isSubmitted && (
                <motion.div 
                  className="success-message"
                  initial={{ opacity: 0, scale: 0.9 }}
                  animate={{ opacity: 1, scale: 1 }}
                  transition={{ duration: 0.3 }}
                >
                  <CheckCircle size={24} strokeWidth={2} className="success-icon" />
                  <span>Вы успешно записались на курс!</span>
                </motion.div>
              )}
            </form>
          </div>
        </motion.section>

        <motion.section 
          className="guarantees-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.2 }}
        >
          <h2 className="section-title">Гарантии</h2>
          <div className="guarantees-grid">
            {guarantees.map((guarantee, index) => {
              const IconComponent = guarantee.icon
              return (
                <motion.div
                  key={guarantee.title}
                  className="guarantee-item"
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.4 }}
                  whileHover={{ x: 3, transition: { duration: 0.2 } }}
                >
                  <div className="guarantee-icon">
                    <IconComponent size={28} strokeWidth={2} color="#2563EB" />
                  </div>
                  <h3 className="guarantee-title">{guarantee.title}</h3>
                  <p className="guarantee-text">{guarantee.text}</p>
                </motion.div>
              )
            })}
          </div>
        </motion.section>

        <motion.section 
          className="instructors-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.3 }}
        >
          <h2 className="section-title">Наши преподаватели</h2>
          <div className="instructors-grid">
            {instructors.map((instructor, index) => (
              <motion.div
                key={instructor.name}
                className="instructor-card"
                initial={{ opacity: 0, scale: 0.95 }}
                animate={{ opacity: 1, scale: 1 }}
                transition={{ delay: index * 0.1, duration: 0.4 }}
                whileHover={{ x: 3, transition: { duration: 0.2 } }}
              >
                <div className="instructor-avatar">
                  <User size={32} strokeWidth={2} />
                </div>
                <h3 className="instructor-name">{instructor.name}</h3>
                <p className="instructor-experience">{instructor.experience} опыта</p>
                <div className="instructor-rating">
                  {Array.from({ length: 5 }, (_, i) => (
                    <Star key={i} size={16} fill="#2563EB" stroke="#2563EB" strokeWidth={2} />
                  ))}
                </div>
                <p className="instructor-specialty">{instructor.specialty}</p>
                <div className="instructor-students">
                  <Users size={16} strokeWidth={2} />
                  <span>{instructor.students} студентов</span>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default Enroll

